using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>07/25/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// IssuedUnit data access layer class.
	///</summary>

	#endregion
	
	public class IssuedUnit
    {
        
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4860"> 
		///		<ExpectedInput>Valid Issued unit unique identifier, Read Only Indicator and update user</ExpectedInput>
		///		<ExpectedOutput>Data table containing issued unit information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4861"> 
		///		<ExpectedInput>empty unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetIssuedUnit
		/// </summary>
		/// <param name="issuedUnitGuid">issuedUnitGUID</param>
		/// <returns>Data Table with Issued Units</returns>
        public static DataTable GetIssuedUnit(System.Guid issuedUnitGuid)
        {
            SqlParameter[] prms =
            {
                new SqlParameter(STOREDPROC.GetIssuedUnit.issuedunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetIssuedUnit.divisioncode, SqlDbType.Char)		
            };

            prms[0].Value = issuedUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode; 

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssuedUnit.StoredProcName, prms);
			return ds.Tables[0];
        }


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7228"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ArrayList of strings</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7229"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetRemoteStorageLocations
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetRemoteStorageLocations()
		{	
			ArrayList locations = new ArrayList();
			DataTable dt = Common.StoredProcedure.GetData(STOREDPROC.GetRemoteStorageLocations.StoredProcName).Tables[0];
			foreach (DataRow dtRow in dt.Rows)
			{
				if (!dtRow.IsNull(Common.VbecsTables.IssuedUnit.RemoteStorageText)) locations.Add(dtRow[Common.VbecsTables.IssuedUnit.RemoteStorageText]);
			}
			return locations;
		}
        
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4862"> 
		///		<ExpectedInput>Valid blood unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table containing issued unit information for a specified blood unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4863"> 
		///		<ExpectedInput>Empty blood unit unique identifier</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return issued unit information for a specified blood unit
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <returns>Data table of issued unit information</returns>
		public static DataTable GetIssuedUnitByBloodUnitGuid(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetIssuedUnitFromGuid.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetIssuedUnitFromGuid.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssuedUnitFromGuid.StoredProcName, prms);
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3728"> 
		///		<ExpectedInput>Valid patient unique identifier</ExpectedInput>
		///		<ExpectedOutput>Data table containing all units issued for a patient</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3729"> 
		///		<ExpectedInput>Empty patient unique identifier</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return all units currently issed to the supplied patient within the division
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <returns>Data table of issued unit information</returns>
		public static DataTable GetIssuedUnitsForPatient(System.Guid patientGuid)
		{
			if (patientGuid==Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("Patient unique identifier").ResString);
			}
						
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetIssuedUnitsForPatient.patientguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetIssuedUnitsForPatient.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = patientGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssuedUnitsForPatient.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Magaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4769"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4770"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return all units currently issed within the division
		/// </summary>
		/// <returns>Data table of issued unit information</returns>
		public static DataTable GetIssuedUnits()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetIssuedUnitsList.divisioncode, SqlDbType.Char)
			};
		
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssuedUnitsList.StoredProcName, prms);
					
			return ds.Tables[0];
		}
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4864"> 
		///		<ExpectedInput>Valid DataTables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4865"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InsertIssuedUnit
		/// </summary>
		/// <param name="issuedUnits">IssuedUnits</param>
		/// <param name="releaseUnits">releasedUnits</param>
		/// <param name="statusUnits">statusUnits</param>
		/// <param name="specimens"></param>
		/// <param name="issueUnitComments"></param> 
		/// <param name="dtWorkloadEvents"></param>
		/// <param name="dtIssueExceptions"></param>
		/// <param name="dtBCEMessages"></param>
		/// <returns>Data table</returns>
		public static bool InsertIssuedUnits(DataTable issuedUnits, 
												DataTable releaseUnits, 
												DataTable statusUnits, 
												DataTable specimens, 
												DataTable issueUnitComments, 
												DataTable dtWorkloadEvents,
												DataTable dtIssueExceptions,
												DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			ArrayList spArray = new ArrayList();
			ArrayList dtArray = new ArrayList();

			spArray.Add(Common.VbecsStoredProcs.InsertIssuedUnit.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.ReleaseOrderedUnitsByBloodUnit.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.UpdateBloodUnitStatus.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.UpdatePatientSpecimen.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.InsertIssuedUnitComment.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.InsertWorkloadEvents.StoredProcName);
			spArray.Add(Common.VbecsStoredProcs.InsertExceptionUnitIssue.StoredProcName);
			if (dtBCEMessages != null)
				spArray.Add(Common.VbecsStoredProcs.HL7InsertMessageLog.StoredProcName);

			dtArray.Add(issuedUnits);
			dtArray.Add(releaseUnits);
			dtArray.Add(statusUnits);
			dtArray.Add(specimens);
			dtArray.Add(issueUnitComments);
			dtArray.Add(dtWorkloadEvents);
			dtArray.Add(dtIssueExceptions);
			if (dtBCEMessages != null)
				dtArray.Add(dtBCEMessages);

			return ((new StoredProcedure()).TransactionalGetValue(spArray,dtArray)==0);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4866"> 
		///		<ExpectedInput>OrderedUnitGUid, Row Versions</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4867"> 
		///		<ExpectedInput>OrderedUnitGuid, Invalid RowVersions</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MarkUnitUnsatisfactory
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="rowVersion"></param>
		/// <param name="unitStatusRowVersion"></param>
		/// <param name="updateFunction"></param>
		/// <param name="dtBCEMessages"></param>
		/// <returns></returns>
		public static bool MarkUnitUnsatisfactory(Guid orderedUnitGuid, 
			byte [] rowVersion, 
			byte [] unitStatusRowVersion,
			Common.UpdateFunction updateFunction,
			DataTable dtBCEMessages)
	{
			//CR 2941, CR 2945 Added dtBCEMessages
			if ( orderedUnitGuid == Guid.Empty || rowVersion == null || unitStatusRowVersion == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns("parameter").ResString);
			}

			DataTable dtUnsatisfactoryUnit = new DataTable();
			dtUnsatisfactoryUnit.Columns.Add(TABLES.OrderedUnit.OrderedUnitGuid, typeof(Guid));
			dtUnsatisfactoryUnit.Columns.Add(TABLES.OrderedUnit.LastUpdateUser);
			dtUnsatisfactoryUnit.Columns.Add(TABLES.OrderedUnit.RowVersion, typeof(byte[]));
			dtUnsatisfactoryUnit.Columns.Add(ARTIFICIAL.UnitStatusRowVersion, typeof(byte[]));
			dtUnsatisfactoryUnit.Columns.Add(TABLES.OrderedUnit.LastUpdateFunctionId, typeof(int));

			dtUnsatisfactoryUnit.Rows.Add(dtUnsatisfactoryUnit.NewRow());
			dtUnsatisfactoryUnit.Rows[0][TABLES.OrderedUnit.OrderedUnitGuid] = orderedUnitGuid;
			dtUnsatisfactoryUnit.Rows[0][TABLES.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dtUnsatisfactoryUnit.Rows[0][TABLES.OrderedUnit.RowVersion] = rowVersion;
			dtUnsatisfactoryUnit.Rows[0][ARTIFICIAL.UnitStatusRowVersion] = unitStatusRowVersion;
			dtUnsatisfactoryUnit.Rows[0][TABLES.OrderedUnit.LastUpdateFunctionId] = updateFunction;

			string [] spArray = {STOREDPROC.CancelOrderedUnit.StoredProcName,
									STOREDPROC.HL7InsertMessageLog.StoredProcName};

			DataTable [] dtArray = {dtUnsatisfactoryUnit,
									   dtBCEMessages};

			return (new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray)==0);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8040"> 
		///		<ExpectedInput>Valid DataTables</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8041"> 
		///		<ExpectedInput>DataTables with invalid row Version</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MarkUnitUnsatisfactory
		/// </summary>
		/// <returns></returns>
		public static bool MarkUnitUnsatisfactory(DataTable dtReturnIssuedUnits,
												DataTable dtStatus,
												DataTable dtOrderedUnit,
												DataTable dtReturnExceptions,
												DataTable dtWorkloadEvent,
												DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			string [] spArray = {STOREDPROC.UpdateIssuedUnit.StoredProcName,
									STOREDPROC.UpdateBloodUnitStatus.StoredProcName,
									STOREDPROC.ReleaseOrderedUnits.StoredProcName,
									STOREDPROC.InsertExceptionReturnIssuedUnit.StoredProcName,
									STOREDPROC.InsertWorkloadEvents.StoredProcName,
									STOREDPROC.HL7InsertMessageLog.StoredProcName};

			DataTable [] dtArray = {Common.Utility.AppendLastUpdateInformation(dtReturnIssuedUnits, Common.UpdateFunction.UC059FrmReturnIssuedUnits),
									   Common.Utility.AppendLastUpdateInformation(dtStatus, Common.UpdateFunction.UC059FrmReturnIssuedUnits), 
									   Common.Utility.AppendLastUpdateInformation(dtOrderedUnit, Common.UpdateFunction.UC059FrmReturnIssuedUnits),
									   dtReturnExceptions,
									   Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, Common.UpdateFunction.UC059FrmReturnIssuedUnits),
									   dtBCEMessages};

			return (new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray)==0);
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4809"> 
		///		<ExpectedInput>DataSet, boolean, UpdateFunctionid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4810"> 
		///		<ExpectedInput>Invalid DataSet</ExpectedInput>
		///		<ExpectedOutput>NullReferenceException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns issued units to blood bank, and also releases ordered units from assignment.
		/// </summary>
		/// <param name="dsSave"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Success indicator</returns>
		public static bool Update(DataSet dsSave, Common.UpdateFunction lastUpdateFunctionId)
		{
			int NumberOfTableToSave = dsSave.Tables.Count;
			int Counter = 0;
			int returnValue = 0;
			StoredProcedure sproc = new StoredProcedure();

			DataTable[] dtArray = new DataTable[NumberOfTableToSave];
			string[] spArray = new string[NumberOfTableToSave];

			foreach (DataTable dtSave in dsSave.Tables)
			{
				if (dtSave.TableName == Common.VbecsTables.IssuedUnit.TableName)
				{
					dtArray[Counter] = Common.Utility.AppendLastUpdateInformation(dtSave,lastUpdateFunctionId);
					spArray[Counter] = STOREDPROC.UpdateIssuedUnit.StoredProcName;

				}
				else if (dtSave.TableName == Common.VbecsTables.ExceptionReturnedUnit.TableName)
				{
					dtArray[Counter] = dtSave;
					spArray[Counter] = STOREDPROC.InsertExceptionReturnIssuedUnit.StoredProcName;
				}
				else if(dtSave.TableName == Common.VbecsTables.WorkloadEvent.TableName)
				{
					dtArray[Counter] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[Counter] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
				}
				//CR 2941, CR 2945 Added MessageLog table
				else if(dtSave.TableName == Common.VbecsTables.MessageLog.TableName)
				{
					dtArray[Counter] = dtSave;
					spArray[Counter] = STOREDPROC.HL7InsertMessageLog.StoredProcName;
				}
				Counter++;
			}

			returnValue = sproc.TransactionalGetValue(spArray, dtArray);

			if(returnValue == 0)
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2125"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of Locations</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2126"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of issue-to locations for the division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetIssueToLocations()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetIssueToLocationList.divisioncode, SqlDbType.Char)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetIssueToLocationList.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2132"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of Physicians</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2134"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of physicians for the division
		/// </summary>
		/// <returns></returns>
		public static DataTable GetPhysicianList()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetPhysicianList.divisioncode, SqlDbType.Char)
			};
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetPhysicianList.StoredProcName, prms);

			return ds.Tables[0];
		}
    }
}
